using System;
using System.Runtime.InteropServices;  /* Provides mappings between C# and native code */

/* This example calls either a Windows or Linux version of the NAG Library based on
 * which OS the example is being executed on*/

namespace NumericalAlgorithmsGroup
{
  class MultipleLinearRegression
    {

        /* NAG routine declarations copied from flcsdnet64.cs & flcsdnet64_linux.cs*/

        /* Both Windows and Linux routine declarations included as the dll is not 
         * loaded unless the routine is called explicitly.
         * The Linux declaration has an underscore appended to differentiate it*/

        [DllImport("NLW6I31DE_nag.dll")] /* Import from DLL, the C# compiler provides a rudimentary check of the signature */
        public static extern void G02EAF(
                                         string MEAN,
                                         string WEIGHT,
                                         ref int N,
                                         ref int M,
                                         [In] double[,] X,
                                         ref int LDX,
                                         [In,Out] byte[,] VNAME,
                                         [In] int[] ISX,
                                         [In] double[] Y,
                                         [In] double[] WT,
                                         ref int NMOD,
                                         [In,Out] byte[,,] MODL,
                                         ref int LDMODL,
                                         [Out] double[] RSS,
                                         [Out] int[] NTERMS,
                                         [Out] int[] MRANK,
                                         [Out] double[] WK,
                                         ref int IFAIL,
                                         long MEANLength,
                                         long WEIGHTLength,
                                         long VNAMELength,
                                         long MODLLength
                                         );

     [DllImport("libnag_nag.so", EntryPoint="g02eaf_")]
        public static extern void G02EAF_(
                                         string MEAN,
                                         string WEIGHT,
                                         ref int N,
                                         ref int M,
                                         [In] double[,] X,
                                         ref int LDX,
                                         [In,Out] byte[,] VNAME,
                                         [In] int[] ISX,
                                         [In] double[] Y,
                                         [In] double[] WT,
                                         ref int NMOD,
                                         [In,Out] byte[,,] MODL,
                                         ref int LDMODL,
                                         [Out] double[] RSS,
                                         [Out] int[] NTERMS,
                                         [Out] int[] MRANK,
                                         [Out] double[] WK,
                                         ref int IFAIL,
                                         long MEANLength,
                                         long WEIGHTLength,
                                         long VNAMELength,
                                         long MODLLength
                                         );


        // Convenience routine copies a string into a byte array
        private static void setName(byte[,] name, int i, string sname)
        {
            for (int j = 0; j < sname.Length; j++)
                name[i, j] = (byte)sname[j];
        }

        public static void Main()
        {
            // We solve the example problem presented in the documentation of
            // routine G02EAF in the NAG Fortran Library Manual.
            int n = 20;
            int m = 6;

            string mean = "M";     // Mean(M) or Zero(Z)
            string weight = "U";   // Unweighted(U) or Weighted(W)

            double[] wt = new double[n];
            for (int i = 0; i < n; i++)
                wt[i] = 1;
            int[] isx = new int[n];
            for (int i = 0; i < n; i++)
                isx[i] = 1;
            isx[0] = 0;

            // Notice that the 2D input array X is stored in transposed form
            // because the NAG Fortran Library expects 2D arrays to be stored
            // by column rather than by row.
            double[,] x = new double[,]
            {
              {0.0, 1125.0, 232.0, 7160.0, 85.9, 8905.0},
              {7.0, 920.0, 268.0, 8804.0, 86.5, 7388.0},
              {15.0, 835.0, 271.0, 8108.0, 85.2, 5348.0},
              {22.0, 1000.0, 237.0, 6370.0, 83.8, 8056.0},
              {29.0, 1150.0, 192.0, 6441.0, 82.1, 6960.0},
              {37.0, 990.0, 202.0, 5154.0, 79.2, 5690.0},
              {44.0, 840.0, 184.0, 5896.0, 81.2, 6932.0},
              {58.0, 650.0, 200.0, 5336.0, 80.6, 5400.0},
              {65.0, 640.0, 180.0, 5041.0, 78.4, 3177.0},
              {72.0, 583.0, 165.0, 5012.0, 79.3, 4461.0},
              {80.0, 570.0, 151.0, 4825.0, 78.7, 3901.0},
              {86.0, 570.0, 171.0, 4391.0, 78.0, 5002.0},
              {93.0, 510.0, 243.0, 4320.0, 72.3, 4665.0},
              {100.0, 555.0, 147.0, 3709.0, 74.9, 4642.0},
              {107.0, 460.0, 286.0, 3969.0, 74.4, 4840.0},
              {122.0, 275.0, 198.0, 3558.0, 72.5, 4479.0},
              {129.0, 510.0, 196.0, 4361.0, 57.7, 4200.0},
              {151.0, 165.0, 210.0, 3301.0, 71.8, 3410.0},
              {171.0, 244.0, 327.0, 2964.0, 72.5, 3360.0},
              {220.0, 79.0, 334.0, 2777.0, 71.9, 2599.0}
            };

            // Notice that array X is transposed to XT transposed because
            // the NAG Fortran library has arrays in column-major order
            double[,] xt = new double[6, 20];
            for (int i = 0; i < n; i++)
                for (int j = 0; j < m; j++)
                    xt[j, i] = x[i, j];

            double[] y = new double[] { 1.5563, 0.8976, 0.7482, 0.7160, 0.3010, 0.3617,
                                        0.1139, 0.1139, -0.2218, -0.1549, 0.0000, 0.0000,
                                        -0.0969, -0.2218, -0.3979, -0.1549, -0.2218, -0.3979,
                                        -0.5229, -0.0458 };

            // Character string array arguments are most easily passed to the NAG Library
            // as byte arrays. Notice that we declare the array with an extra
            // dimension, which is the length of each string in the array (in
            // this case, 3).
            byte[,] name = new byte[n, 3];
            setName(name, 0, "DAY");
            setName(name, 1, "BOD");
            setName(name, 2, "TKN");
            setName(name, 3, "TS ");
            setName(name, 4, "TVS");
            setName(name, 5, "COD");

            int ldm = (int)Math.Pow(2, 5);
            // We give the 2D character string array "model" an extra
            // dimension, the length of the character strings, which
            // must match that of argument "name" - in this case, 3.
            byte[,,] model = new byte[m, ldm, 3];
            int[] nterms = new int[ldm];
            double[] rss = new double[ldm];
            int[] mrank = new int[ldm];
            double[] wk = new double[n * (m + 1)];
            int nmod = 0;

            // Calculate residual sum-of-squares for all possible linear regressions
            // of the set of independent variables in X. Notice that we pass the
            // hidden length arguments of the character arrays name and model,
            // immediately after the arguments themselves.
            int ifail = 1;
            if (RuntimeInformation.IsOSPlatform(OSPlatform.Windows))
            {
                G02EAF(mean, weight,
                       ref n, ref m, xt, ref n, name, isx, y, wt,
                       ref nmod, model, ref ldm, rss, nterms, mrank, wk, ref ifail,
                       mean.Length, weight.Length, name.GetLength(1), model.GetLength(2));
            }
            else if (RuntimeInformation.IsOSPlatform(OSPlatform.Linux))
            {
                G02EAF_(mean, weight,
                       ref n, ref m, xt, ref n, name, isx, y, wt,
                       ref nmod, model, ref ldm, rss, nterms, mrank, wk, ref ifail,
                       mean.Length, weight.Length, name.GetLength(1), model.GetLength(2));
            }
            else
            {
                Console.WriteLine("Unsupported OS Platform");
                return;
            }

            if (ifail != 0)
            {
                Console.WriteLine("ifail = {0}", ifail);
            }
            else
            {
                // Show the results
                Console.WriteLine("G02EAF Example Program Results");
                Console.WriteLine();
                Console.WriteLine(" Number of       RSS       RANK   MODEL");
                Console.WriteLine("parameters");
                for (int i = 0; i < nmod; i++)
                {
                    int ii = nterms[i];
                    Console.Write("{0, 6}   {1, 12:f4}   {2, 6}    ", ii, rss[i], mrank[i]);
                    for (int j = 0; j < ii; j++)
                    {
                        Console.Write("{0}{1}{2}  ",
                                      (char)model[j, i, 0], (char)model[j, i, 1], (char)model[j, i, 2]);
                    }
                    Console.WriteLine();
                }
            }

        }
    }
}
