using System;
using System.Text;
using System.Runtime.InteropServices;/* Provides mappings between C# and native code */

/* This example calls either a Windows or Linux version of the NAG Library based on
 * which OS the example is being executed on*/

namespace NumericalAlgorithmsGroup
{
  class LUSolve
    {

        /* NAG routine declarations copied from flcsdnet64.cs & flcsdnet64_linux.cs*/

        /* Both Windows and Linux routine declarations included as the dll is not 
         * loaded unless the routine is called explicitly.
         * The Linux declaration has an underscore appended to differentiate it*/

        [StructLayout(LayoutKind.Sequential)]/* Force sequential mapping */
	public struct Complex
	{
	  public double Real_Part;
	  public double Imag_Part;
	};
      
      [DllImport("NLW6I31DE_nag.dll")] /* Import from DLL, the C# compiler provides a rudimentary check of the signature */
        public static extern void ZGESVX(
                                         string FACT, string TRANS, ref int N,
                                         ref int NRHS, [In,Out] Complex[,] A, ref int LDA,
                                         [In,Out] Complex[,] AF, ref int LDAF, [In,Out] int[] IPIV,
                                         StringBuilder EQUED, [In,Out] double[] R, [In,Out] double[] C,
                                         [In,Out] Complex[,] B, ref int LDB, [Out] Complex[,] X,
                                         ref int LDX, ref double RCOND, [Out] double[] FERR,
                                         [Out] double[] BERR, [Out] Complex[] WORK, [Out] double[] RWORK,
                                         ref int INFO,
                                         long FACTLength,
                                         long TRANSLength,
                                         long EQUEDLength
                                         );

      [DllImport("libnag_nag.so", EntryPoint="zgesvx_")]
        public static extern void ZGESVX_(
                                         string FACT, string TRANS, ref int N,
                                         ref int NRHS, [In,Out] Complex[,] A, ref int LDA,
                                         [In,Out] Complex[,] AF, ref int LDAF, [In,Out] int[] IPIV,
                                         StringBuilder EQUED, [In,Out] double[] R, [In,Out] double[] C,
                                         [In,Out] Complex[,] B, ref int LDB, [Out] Complex[,] X,
                                         ref int LDX, ref double RCOND, [Out] double[] FERR,
                                         [Out] double[] BERR, [Out] Complex[] WORK, [Out] double[] RWORK,
                                         ref int INFO,
                                         long FACTLength,
                                         long TRANSLength,
                                         long EQUEDLength
                                         );

      [DllImport("NLW6I31DE_nag.dll")]
        public static extern void X04DAF(
                                         string MATRIX,
                                         string DIAG,
                                         ref int M,
                                         ref int N,
                                         [In] Complex[,] A,
                                         ref int LDA,
                                         string TITLE,
                                         ref int IFAIL,
                                         long MATRIXLength,
                                         long DIAGLength,
                                         long TITLELength
                                         );

      [DllImport("libnag_nag.so", EntryPoint="x04daf_")]
        public static extern void X04DAF_(
                                         string MATRIX,
                                         string DIAG,
                                         ref int M,
                                         ref int N,
                                         [In] Complex[,] A,
                                         ref int LDA,
                                         string TITLE,
                                         ref int IFAIL,
                                         long MATRIXLength,
                                         long DIAGLength,
                                         long TITLELength
                                         );

        public static void Main()
        {
            /* Scalars */
            double rcond = 0.0;
            int i, ifail, info = 99, j, n, nrhs;

            /* Character args */
            StringBuilder equed = new StringBuilder("X");
            /* The StringBuilder type is used as equed is both input and output.
               The String type is immutable.
            */

            string Complexadata = "-1.34,2.55,0.28,3.17,-6.39,-2.20,0.72,-0.92,-1.70,-14.10,33.10,-1.50,-1.50,13.40,12.90,13.80,-3.29,-2.39,-1.91,4.42,-0.14,-1.35,1.72,1.35,2.41,0.39,-0.56,1.47,-0.83,-0.69,-1.96,0.67";
            string Complexbdata = "26.26,51.78,31.32,-6.70,64.30,-86.80,158.60,-14.20,-5.75,25.31,-2.15,30.19,1.16,2.57,-2.56,7.55";
            Console.WriteLine("F07APF Example Program Results\n");
            n = 4;
            nrhs = 2;
            Complex[,] a = new Complex[n, n];
            Complex[,] at = new Complex[n, n];
            Complex[,] af = new Complex[n, n];
            Complex[,] b = new Complex[n, nrhs];
            Complex[,] bt = new Complex[nrhs, n];
            double[] berr = new double[nrhs];
            double[] c = new double[n];
            double[] ferr = new double[nrhs];
            double[] r = new double[n];
            Complex[] work = new Complex[4 * n];
            Complex[,] x = new Complex[nrhs, n];

            int[] ipiv = new int[n];
            double[] rwork = new double[2 * n];

            /* Read data for a*/
            string[] Complexdataarray = Complexadata.Split(',');
            int index = 0;
            for (i = 0; i < n; i++)
            {
                for (j = 0; j < n; j++)
                {
                    a[i, j].Real_Part = double.Parse(Complexdataarray[index++]);
                    a[i, j].Imag_Part = double.Parse(Complexdataarray[index++]);
                }
            }

            /* Transpose A */
            for (i = 0; i < n; ++i)
            {
                for (j = 0; j < n; ++j)
                {
                    at[j, i] = a[i, j];
                }
            }

            /* Read data for b */
            Complexdataarray = Complexbdata.Split(',');
            index = 0;
            for (i = 0; i < n; i++)
            {
                for (j = 0; j < nrhs; j++)
                {
                    b[i, j].Real_Part = double.Parse(Complexdataarray[index++]);
                    b[i, j].Imag_Part = double.Parse(Complexdataarray[index++]);
                }
            }

            /* Transpose b */
            for (i = 0; i < n; ++i)
            {
                for (j = 0; j < nrhs; ++j)
                {
                    bt[j, i] = b[i, j]; /* bt_ref macros carries out the transform */
                }
            }

            /* Solve the equations AX = B for X */

            if (RuntimeInformation.IsOSPlatform(OSPlatform.Windows))
            {
                ZGESVX("Equilibration", "No transpose", ref n, ref nrhs, at, ref n,
                       af, ref n, ipiv, equed, r, c, bt, ref n, x, ref n, ref rcond, ferr,
                       berr, work, rwork, ref info, "Equilibration".Length, "No transpose".Length,
                       equed.ToString().Length);
            }
            else if (RuntimeInformation.IsOSPlatform(OSPlatform.Linux))
            {
                ZGESVX_("Equilibration", "No transpose", ref n, ref nrhs, at, ref n,
                       af, ref n, ipiv, equed, r, c, bt, ref n, x, ref n, ref rcond, ferr,
                       berr, work, rwork, ref info, "Equilibration".Length, "No transpose".Length,
                       equed.ToString().Length);
            }
            else
            {
                Console.WriteLine("Unsupported OS Platform");
                return;
            }

            if (info == 0 || info == n + 1)
            {

                /*  Print solution, error bounds, condition number, the form */
                /*  of equilibration and the pivot growth factor */

                /* x is returned from Fortran and so is 'transposed' already */
                ifail = 1;
                if (RuntimeInformation.IsOSPlatform(OSPlatform.Windows))
                {
                    X04DAF("General", " ", ref n, ref nrhs, x, ref n, "Solution(s)",
                        ref ifail, "General".Length, " ".Length, "Solution(s)".Length);
                }
                else if (RuntimeInformation.IsOSPlatform(OSPlatform.Linux))
                {
	            X04DAF_("General", " ", ref n, ref nrhs, x, ref n, "Solution(s)",
                        ref ifail, "General".Length, " ".Length, "Solution(s)".Length);
                }
                if (ifail != 0)
                {
                    Console.WriteLine("ifail = {0}", ifail);
                }
                else
                {
                    Console.WriteLine("\nBackward errors (machine-dependent");

                    for (j = 1; j <= nrhs; ++j)
                    {
                        Console.WriteLine("{0, 11:e1}{1}", berr[j - 1], j % 7 == 0 || j == nrhs ? "\n" : "");
                    }

                    Console.WriteLine("Estimated forward error bounds (machine-dependent)");
                    for (j = 1; j <= nrhs; ++j)
                    {
                        Console.WriteLine("{0, 11:e1}{1}", ferr[j - 1], j % 7 == 0 || j == nrhs ? "\n" : "");
                    }

                    /* Console.WriteLine("equed = {0}", equed.ToString());*/
                    if (equed.ToString() == "N")
                    {
                        Console.WriteLine("A has not been equilibrated");
                    }
                    else if (equed.ToString() == "R")
                    {
                        Console.WriteLine("A has been row scaled as diag(R)*A");
                    }
                    else if (equed.ToString() == "C")
                    {
                        Console.WriteLine("A has been column scaled as A*diag(C)");
                    }
                    else if (equed.ToString() == "B")
                    {
                        Console.WriteLine("A has been row and column scaled as diag(R)*A*diag(C)");
                    }

                    Console.WriteLine("\nReciprocal condition number estimate of scaled matrix");
                    Console.WriteLine("{0, 11:e1}", rcond);

                    Console.WriteLine("\nEstimate of reciprocal pivot growth factor");
                    Console.WriteLine("{0, 11:e1}", rwork[0]);
                    if (info == n + 1)
                    {
                        Console.WriteLine("The matrix A is singular to working precision");
                    }
                }
            }
            else
            {
                Console.WriteLine("The ( {0}) element of the factor U is zero", info);
            }
            return;
        }
    }
}
