using System;
using System.Text;
using System.Runtime.InteropServices; /* Provides mappings between C# and native code */

/* This example calls either a Windows or Linux version of the NAG Library based on
 * which OS the example is being executed on*/

namespace NumericalAlgorithmsGroup
{
    class LUSolve
    {

        /* NAG routine declarations copied from flcsdnet64.cs & flcsdnet64_linux.cs*/

        /* Both Windows and Linux routine declarations included as the dll is not 
         * loaded unless the routine is called explicitly.
         * The Linux declaration has an underscore appended to differentiate it*/

        [DllImport("NLW6I31DE_nag.dll")] /* Import from DLL, the C# compiler provides a rudimentary check of the signature */
        public static extern void DGESVX(
                                         string FACT, string TRANS, ref int N,
                                         ref int NRHS, [In,Out] double[,] A, ref int LDA,
                                         [In,Out] double[,] AF, ref int LDAF, [In,Out] int[] IPIV,
                                         StringBuilder EQUED, [In,Out] double[] R, [In,Out] double[] C,
                                         [In,Out] double[,] B, ref int LDB, [Out] double[,] X,
                                         ref int LDX, ref double RCOND, [Out] double[] FERR,
                                         [Out] double[] BERR, [Out] double[] WORK, [Out] int[] IWORK,
                                         ref int INFO,
                                         long FACTLength,
                                         long TRANSLength,
                                         long EQUEDLength
                                         );

        [DllImport("libnag_nag.so", EntryPoint="dgesvx_")]
        public static extern void DGESVX_(
                                         string FACT, string TRANS, ref int N,
                                         ref int NRHS, [In,Out] double[,] A, ref int LDA,
                                         [In,Out] double[,] AF, ref int LDAF, [In,Out] int[] IPIV,
                                         StringBuilder EQUED, [In,Out] double[] R, [In,Out] double[] C,
                                         [In,Out] double[,] B, ref int LDB, [Out] double[,] X,
                                         ref int LDX, ref double RCOND, [Out] double[] FERR,
                                         [Out] double[] BERR, [Out] double[] WORK, [Out] int[] IWORK,
                                         ref int INFO,
                                         long FACTLength,
                                         long TRANSLength,
                                         long EQUEDLength
                                         );


        [DllImport("NLW6I31DE_nag.dll")]
        public static extern void X04CAF(
                                         string MATRIX,
                                         string DIAG,
                                         ref int M,
                                         ref int N,
                                         [In] double[,] A,
                                         ref int LDA,
                                         string TITLE,
                                         ref int IFAIL,
                                         long MATRIXLength,
                                         long DIAGLength,
                                         long TITLELength
                                         );

        [DllImport("libnag_nag.so", EntryPoint="x04caf_")]
        public static extern void X04CAF_(
                                         string MATRIX,
                                         string DIAG,
                                         ref int M,
                                         ref int N,
                                         [In] double[,] A,
                                         ref int LDA,
                                         string TITLE,
                                         ref int IFAIL,
                                         long MATRIXLength,
                                         long DIAGLength,
                                         long TITLELength
                                         );

        public static void Main()
        {
            /* Scalars */
            double rcond = 0.0;
            int i, ifail, info = 0, j, n, nrhs;

            /* Character args */
            StringBuilder equed = new StringBuilder("X");
            /* The StringBuilder type is used as equed is both input and output.
               The String type is immutable.
            */

            Console.WriteLine("F07ABF Example Program Results\n");
            n = 4;
            nrhs = 2;
            double[,] a = new double[4, 4] {
                {1.80, 2.88, 2.05, -0.89},
                {525.00, -295.00, -95.00, -380.00},
                {1.58, -2.69, -2.90, -1.04},
                {-1.11, -0.66, -0.59, 0.80}
            };
            double[,] at = new double[n, n];
            double[,] af = new double[n, n];
            double[,] b = new double[4, 2]{
                {9.52, 18.47},
                {2435.00, 225.00},
                {0.77, -13.28},
                {-6.22, -6.21}
            };
            double[,] bt = new double[nrhs, n];
            double[] berr = new double[nrhs];
            double[] c = new double[n];
            double[] ferr = new double[nrhs];
            double[] r = new double[n];
            double[] work = new double[4 * n];
            double[,] x = new double[n, nrhs];
            int[] ipiv = new int[n];
            int[] iwork = new int[n];

            /* Transpose A */
            for (i = 0; i < n; ++i)
            {
                for (j = 0; j < n; ++j)
                {
                    at[j, i] = a[i, j];
                }
            }

            /* Transpose b */
            for (i = 0; i < n; ++i)
            {
                for (j = 0; j < nrhs; ++j)
                {
                    bt[j, i] = b[i, j];
                }
            }

            /* Solve the equations AX = B for X */

            if (RuntimeInformation.IsOSPlatform(OSPlatform.Windows))
            {
                DGESVX("Equilibration", "No transpose", ref n, ref nrhs, at, ref n,
                       af, ref n, ipiv, equed, r, c, bt, ref n, x, ref n, ref rcond, ferr,
                       berr, work, iwork, ref info,
                       "Equilibration".Length, "No transpose".Length, equed.ToString().Length);
            }
            else if (RuntimeInformation.IsOSPlatform(OSPlatform.Linux))
            {
                DGESVX_("Equilibration", "No transpose", ref n, ref nrhs, at, ref n,
                       af, ref n, ipiv, equed, r, c, bt, ref n, x, ref n, ref rcond, ferr,
                       berr, work, iwork, ref info,
                       "Equilibration".Length, "No transpose".Length, equed.ToString().Length);
            }
            else
            {
                Console.WriteLine("Unsupported OS Platform");
                return;
            }

            if (info == 0 || info == n + 1)
            {
                /* Print solution, error bounds, condition number, the form */
                /* of equilibration and the pivot growth factor */

                ifail = 1;
                /* x is returned from Fortran and so is 'transposed' already */
                if (RuntimeInformation.IsOSPlatform(OSPlatform.Windows))
                {
                    X04CAF("General", " ", ref n, ref nrhs, x, ref n, "Solution(s)",
                           ref ifail, "General".Length, " ".Length, "Solution(s)".Length);
                }
                else if (RuntimeInformation.IsOSPlatform(OSPlatform.Linux))
                {
                    X04CAF_("General", " ", ref n, ref nrhs, x, ref n, "Solution(s)",
                           ref ifail, "General".Length, " ".Length, "Solution(s)".Length);
                }
                if (ifail != 0)
                {
                    Console.WriteLine("ifail = {0}", ifail);
                }
                else
                {
                    Console.WriteLine("\nBackward errors (machine-dependent");

                    for (j = 1; j <= nrhs; ++j)
                    {
                        Console.WriteLine("{0, 11:e1}{1}", berr[j - 1], j % 7 == 0 || j == nrhs ? "\n" : "");
                    }

                    Console.WriteLine("Estimated forward error bounds (machine-dependent)");
                    for (j = 1; j <= nrhs; ++j)
                    {
                        Console.WriteLine("{0, 11:e1}{1}", ferr[j - 1], j % 7 == 0 || j == nrhs ? "\n" : "");
                    }

                    if (equed.ToString() == "N")
                    {
                        Console.WriteLine("A has not been equilibrated");
                    }
                    else if (equed.ToString() == "R")
                    {
                        Console.WriteLine("A has been row scaled as diag(R)*A");
                    }
                    else if (equed.ToString() == "C")
                    {
                        Console.WriteLine("A has been column scaled as A*diag(C)");
                    }
                    else if (equed.ToString() == "B")
                    {
                        Console.WriteLine("A has been row and column scaled as diag(R)*A*diag(C)");
                    }

                    Console.WriteLine("\nReciprocal condition number estimate of scaled matrix");
                    Console.WriteLine("{0, 11:e1}", rcond);
                    Console.WriteLine();

                    Console.WriteLine("Estimate of reciprocal pivot growth factor");
                    Console.WriteLine("{0, 11:e1}", work[0]);
                    if (info == n + 1)
                    {
                        Console.WriteLine("The matrix A is singular to working precision");
                    }
                }
            }
            else
            {
                Console.WriteLine("The ( {0}) element of the factor U is zero", info);
            }
            return;
        }
    }
}
