using System;
using System.Runtime.InteropServices;/* Provides mappings between C# and native code */

/* This example calls either a Windows or Linux version of the NAG Library based on
 * which OS the example is being executed on*/

namespace NumericalAlgorithmsGroup
{
    class OptNlp
    {

        /* NAG routine declarations copied from flcsdnet64.cs & flcsdnet64_linux.cs*/


        /* Both Windows and Linux routine declarations included as the dll is not 
         * loaded unless the routine is called explicitly.
         * The Linux declaration has an underscore appended to differentiate it*/

        /* Delegate objects, note that:
            1) the scalar parameters are passed by 'ref'
            2) IntPtr is equivalent to the C void pointer. During a callback Fortran passes a void pointer
               as the parameter to C#. The memory pointed to by the IntPtr is reclaimed using Marshal.copy
        */

        public delegate void E04UCA_CONFUN_DELEGATE(
                                                  ref int MODE,
                                                  ref int NCNLN,
                                                  ref int N,
                                                  ref int LDCJ,
                                                  IntPtr NEEDC_iptr,
                                                  IntPtr X_rptr,
                                                  IntPtr C_rptr,
                                                  IntPtr CJAC_rptr,
                                                  ref int NSTATE,
                                                  IntPtr IUSER_iptr,
                                                  IntPtr RUSER_rptr
                                                  );

      public delegate void E04UCA_OBJFUN_DELEGATE(
                                                  ref int MODE,
                                                  ref int N,
                                                  IntPtr X_rptr,
                                                  ref double OBJF,
                                                  IntPtr OBJGRD_rptr,
                                                  ref int NSTATE,
                                                  IntPtr IUSER_iptr,
                                                  IntPtr RUSER_rptr
                                                  );

      [DllImport("NLW6I31DE_nag.dll")] /* Import from DLL, the C# compiler provides a rudimentary check of the signature */
        public static extern void E04UCA(
                                         ref int N, ref int NCLIN, ref int NCNLN,
                                         ref int LDA, ref int LDCJ, ref int LDR,
                                         [In] double[,] A, [In] double[] BL, [In] double[] BU,
                                         E04UCA_CONFUN_DELEGATE CONFUN, E04UCA_OBJFUN_DELEGATE OBJFUN, ref int ITER,
                                         [In,Out] int[] ISTATE, [Out] double[] C, [In,Out] double[,] CJAC,
                                         [In,Out] double[] CLAMDA, ref double OBJF, [Out] double[] OBJGRD,
                                         [In,Out] double[,] R, [In,Out] double[] X, [Out] int[] IWORK,
                                         ref int LIWORK, [Out] double[] WORK, ref int LWORK,
                                         [In,Out] int[] IUSER, [In,Out] double[] RUSER, [In,Out] int[] LWSAV,
                                         [In,Out] int[] IWSAV, [In,Out] double[] RWSAV, ref int IFAIL
                                         );

      [DllImport("libnag_nag.so", EntryPoint="e04uca_")]
        public static extern void E04UCA_(
                                         ref int N, ref int NCLIN, ref int NCNLN,
                                         ref int LDA, ref int LDCJ, ref int LDR,
                                         [In] double[,] A, [In] double[] BL, [In] double[] BU,
                                         E04UCA_CONFUN_DELEGATE CONFUN, E04UCA_OBJFUN_DELEGATE OBJFUN, ref int ITER,
                                         [In,Out] int[] ISTATE, [Out] double[] C, [In,Out] double[,] CJAC,
                                         [In,Out] double[] CLAMDA, ref double OBJF, [Out] double[] OBJGRD,
                                         [In,Out] double[,] R, [In,Out] double[] X, [Out] int[] IWORK,
                                         ref int LIWORK, [Out] double[] WORK, ref int LWORK,
                                         [In,Out] int[] IUSER, [In,Out] double[] RUSER, [In,Out] int[] LWSAV,
                                         [In,Out] int[] IWSAV, [In,Out] double[] RWSAV, ref int IFAIL
                                         );

      [DllImport("NLW6I31DE_nag.dll")]
        public static extern void E04WBF(
                                         string RNAME,
                                         [In,Out] byte[,] CWSAV,
                                         ref int LCWSAV,
                                         [Out] int[] LWSAV,
                                         ref int LLWSAV,
                                         [Out] int[] IWSAV,
                                         ref int LIWSAV,
                                         [Out] double[] RWSAV,
                                         ref int LRWSAV,
                                         ref int IFAIL,
                                         long RNAMELength,
                                         long CWSAVLength
                                         );

      [DllImport("libnag_nag.so", EntryPoint="e04wbf_")]
        public static extern void E04WBF_(
                                         string RNAME,
                                         [In,Out] byte[,] CWSAV,
                                         ref int LCWSAV,
                                         [Out] int[] LWSAV,
                                         ref int LLWSAV,
                                         [Out] int[] IWSAV,
                                         ref int LIWSAV,
                                         [Out] double[] RWSAV,
                                         ref int LRWSAV,
                                         ref int IFAIL,
                                         long RNAMELength,
                                         long CWSAVLength
                                         );

      [DllImport("NLW6I31DE_nag.dll")]
        public static extern void E04UEA(
                                         string STR,
                                         [In,Out] int[] LWSAV,
                                         [In,Out] int[] IWSAV,
                                         [In,Out] double[] RWSAV,
                                         ref int INFORM,
                                         long STRLength
                                         );

      [DllImport("libnag_nag.so", EntryPoint="e04uea_")]
        public static extern void E04UEA_(
                                         string STR,
                                         [In,Out] int[] LWSAV,
                                         [In,Out] int[] IWSAV,
                                         [In,Out] double[] RWSAV,
                                         ref int INFORM,
                                         long STRLength
                                         );


        public static void Main()
        {
            /* Create delegate objects */
            E04UCA_CONFUN_DELEGATE Confun = new E04UCA_CONFUN_DELEGATE(confunex);
            E04UCA_OBJFUN_DELEGATE Objfun = new E04UCA_OBJFUN_DELEGATE(objfunex);
            int n = 4;
            int nclin = 1;
            int ncnln = 2;
            int totalvars = n + nclin + ncnln;
            double[,] a = new double[nclin, n];
            double[] bl = new double[totalvars];
            double[] bu = new double[totalvars];
            int iter = 0;
            int[] istate = new int[totalvars];
            double[] c = new double[ncnln];
            double[,] cjac = new double[ncnln, n];
            double[] clamda = new double[totalvars];
            double objf = 0.0;
            double[] objgrd = new double[n];
            double[,] r = new double[n, n];
            double[] x = new double[n];
            int liwork = 3 * n + nclin + 2 * ncnln;
            int[] iwork = new int[liwork];
            int lwork;
            if (ncnln == 0 && nclin == 0)
            {
                lwork = 20 * n;
            }
            else if (ncnln == 0 && nclin > 0)
            {
                lwork = 2 * n * n + 20 * n + 11 * nclin;
            }
            else
            {
                lwork = 2 * n * n + n * nclin + 2 * n * ncnln + 20 * n + 11 * nclin + 21 * ncnln;
            }
            double[] work = new double[lwork];
            int i, j;

            //   Console.WriteLine("in main program, calling library via interface");
            for (i = 0; i < nclin; ++i)
            {
                for (j = 0; j < n; ++j)
                {
                    a[i, j] = 1.0;
                }
            }
            bl[0] = 1.0;
            bl[1] = 1.0;
            bl[2] = 1.0;
            bl[3] = 1.0;
            bl[4] = -1.0E+25;
            bl[5] = -1.0E+25;
            bl[6] = 25.0;

            bu[0] = 5.0;
            bu[1] = 5.0;
            bu[2] = 5.0;
            bu[3] = 5.0;
            bu[4] = 20.0;
            bu[5] = 40.0;
            bu[6] = 1.0E+25;

            x[0] = 1.0;
            x[1] = 5.0;
            x[2] = 5.0;
            x[3] = 1.0;

            string e04name = new string(' ', 7);
            e04name = "E04UCA";
            int lcwsav = 1;       // number of strings
            int lencwsav = 80;    // length of each string ("Character (80)" in Fortran)
            byte[,] cwsav = new byte[lcwsav, lencwsav];
            int llwsav = 120;
            int[] lwsav = new int[llwsav];
            int liwsav = 610;
            int[] iwsav = new int[liwsav];
            int lrwsav = 475;
            double[] rwsav = new double[lrwsav];
            int ifail = 1;
            int inform = 0;
            int lene04name = e04name.Length;

            if (RuntimeInformation.IsOSPlatform(OSPlatform.Windows))
            {
                E04WBF(e04name, cwsav, ref lcwsav, lwsav, ref llwsav, iwsav, ref liwsav, rwsav, ref lrwsav, ref ifail, lene04name, lencwsav);
            }
            else if (RuntimeInformation.IsOSPlatform(OSPlatform.Linux))
            {
                E04WBF_(e04name, cwsav, ref lcwsav, lwsav, ref llwsav, iwsav, ref liwsav, rwsav, ref lrwsav, ref ifail, lene04name, lencwsav);
            }
            else
            {
                Console.WriteLine("Unsupported OS Platform");
                return;
            }
            if (ifail != 0)
            {
                Console.WriteLine("E04WBF: ifail = {0}", ifail);
            }
            double[] ruser = new double[1];
            int[] iuser = new int[1];
            if (RuntimeInformation.IsOSPlatform(OSPlatform.Windows))
            {
                E04UEA("Print Level = 1", lwsav, iwsav, rwsav, ref inform, "Print Level = 1".Length);
            } 
            else if (RuntimeInformation.IsOSPlatform(OSPlatform.Linux))
            {
                E04UEA_("Print Level = 1", lwsav, iwsav, rwsav, ref ifail, "Print Level = 1".Length);
            }
            ifail = 1;
            if (RuntimeInformation.IsOSPlatform(OSPlatform.Windows))
            {
                E04UCA(ref n, ref nclin, ref ncnln, ref nclin, ref ncnln, ref n, a, bl, bu, Confun,
                       Objfun, ref iter, istate, c, cjac, clamda, ref objf, objgrd, r, x, iwork,
                       ref liwork, work, ref lwork, iuser, ruser, lwsav, iwsav, rwsav, ref ifail);
            }
            else if (RuntimeInformation.IsOSPlatform(OSPlatform.Linux))
            {
                E04UCA_(ref n, ref nclin, ref ncnln, ref nclin, ref ncnln, ref n, a, bl, bu, Confun,
                       Objfun, ref iter, istate, c, cjac, clamda, ref objf, objgrd, r, x, iwork,
                       ref liwork, work, ref lwork, iuser, ruser, lwsav, iwsav, rwsav, ref ifail);
            }
            if (ifail != 0)
            {
                Console.WriteLine("E04UCA: ifail = {0}", ifail);
            }
        }

        public static void objfunex(ref int mode, ref int n, IntPtr x_ptr,
                                    ref double objf, IntPtr objgrd_ptr, ref int nstate,
                                    IntPtr iuser, IntPtr ruser)
        {
            /* Function Body */
            double[] x = new double[n];
            Marshal.Copy(x_ptr, x, 0, n);
            double[] objgrd = new double[n];
            if (mode == 0 || mode == 2)
            {
                objf = x[0] * x[3] * (x[0] + x[1] + x[2]) + x[2];
            }

            if (mode == 1 || mode == 2)
            {
                objgrd[0] = x[3] * (x[0] * 2.0 + x[1] + x[2]);
                objgrd[1] = x[0] * x[3];
                objgrd[2] = x[0] * x[3] + 1.0;
                objgrd[3] = x[0] * (x[0] + x[1] + x[2]);
            }
            Marshal.Copy(objgrd, 0, objgrd_ptr, n);
        }

        public static void confunex(ref int mode, ref int ncnln, ref int n,
                                    ref int ldcj, IntPtr needc_ptr, IntPtr x_ptr,
                                    IntPtr c_ptr, IntPtr cjac_ptr, ref int nstate,
                                    IntPtr iuser_ptr, IntPtr user_ptr)
        {
            int[] needc = new int[ncnln];
            Marshal.Copy(needc_ptr, needc, 0, ncnln);
            double[] x = new double[n];
            Marshal.Copy(x_ptr, x, 0, n);
            double[] c = new double[ncnln];
            Marshal.Copy(c_ptr, c, 0, ncnln);
            double[] cjac = new double[ldcj * n]; /* Marshal.copy does not allow the type of cjac to be [,] */
            Marshal.Copy(cjac_ptr, cjac, 0, ldcj * n);
            if (nstate == 1)
            {
                /*        First call to CONFUN.  Set all Jacobian elements to zero. */
                /*        Note that this will only work when 'Derivative Level = 3' */
                /*        (the default; see Section 11.2). */
                for (int j = 0; j < n; ++j)
                {
                    for (int i = 0; i < ncnln; ++i)
                    {
                        /* Note the expression required to access the elements of sim in column major order*/
                        cjac[j * ncnln + i] = 0.0;
                    }
                }
            }

            if (needc[0] > 0)
            {
                if (mode == 0 || mode == 2)
                {
                    c[0] = x[0] * x[0] + x[1] * x[1] + x[2] * x[2] + x[3] * x[3];
                }
                if (mode == 1 || mode == 2)
                {
                    cjac[0] = x[0] * 2.0;
                    cjac[2] = x[1] * 2.0;
                    cjac[4] = x[2] * 2.0;
                    cjac[6] = x[3] * 2.0;
                }
            }

            if (needc[1] > 0)
            {
                if (mode == 0 || mode == 2)
                {
                    c[1] = x[0] * x[1] * x[2] * x[3];
                }
                if (mode == 1 || mode == 2)
                {
                    cjac[1] = x[1] * x[2] * x[3];
                    cjac[3] = x[0] * x[2] * x[3];
                    cjac[5] = x[0] * x[1] * x[3];
                    cjac[7] = x[0] * x[1] * x[2];
                }

            }
            /* cjac and C are output arrays hence the need to copy back */
            Marshal.Copy(cjac, 0, cjac_ptr, ldcj * n);
            Marshal.Copy(c, 0, c_ptr, ncnln);
        }
    }
}
