using System;
using System.Runtime.InteropServices; /* Provides mappings between C# and native code */

/* This example calls either a Windows or Linux version of the NAG Library based on
 * which OS the example is being executed on*/

namespace NumericalAlgorithmsGroup
{
  class Simplex
    {
        /* NAG routine declarations copied from flcsdnet64.cs & flcsdnet64_linux.cs*/

        /* Both Windows and Linux routine declarations included as the dll is not 
         * loaded unless the routine is called explicitly.
         * The Linux declaration has an underscore appended to differentiate it*/

        /* Delegate object, note that:
            1) the scalar parameters are passed by 'ref' 
            2) IntPtr is equivalent to the C void pointer. During a callback Fortran passes a void pointer
               as the parameter to C#. The memory pointed to by the IntPtr is reclaimed using Marshal.copy
        */

        public delegate void E04CBF_FUNCT_DELEGATE(
                                                 ref int N,
                                                 IntPtr XC_rptr,
                                                 ref double FC,
                                                 IntPtr IUSER_iptr,
                                                 IntPtr RUSER_rptr
                                                 );

      public delegate void E04CBF_MONIT_DELEGATE(
                                                 ref double FMIN,
                                                 ref double FMAX,
                                                 IntPtr SIM_rptr,
                                                 ref int N,
                                                 ref int NCALL,
                                                 ref double SERROR,
                                                 ref double VRATIO,
                                                 IntPtr IUSER_iptr,
                                                 IntPtr RUSER_rptr
                                                 );

      [DllImport("NLW6I31DE_nag.dll")]
        public static extern void E04CBF(
                                         ref int N,
                                         [In,Out] double[] X,
                                         ref double F,
                                         ref double TOLF,
                                         ref double TOLX,
                                         E04CBF_FUNCT_DELEGATE FUNCT,
                                         E04CBF_MONIT_DELEGATE MONIT,
                                         ref int MAXCAL,
                                         [In,Out] int[] IUSER,
                                         [In,Out] double[] RUSER,
                                         ref int IFAIL
                                         );

     [DllImport("libnag_nag.so", EntryPoint="e04cbf_")]
        public static extern void E04CBF_(
                                         ref int N,
                                         [In,Out] double[] X,
                                         ref double F,
                                         ref double TOLF,
                                         ref double TOLX,
                                         E04CBF_FUNCT_DELEGATE FUNCT,
                                         E04CBF_MONIT_DELEGATE MONIT,
                                         ref int MAXCAL,
                                         [In,Out] int[] IUSER,
                                         [In,Out] double[] RUSER,
                                         ref int IFAIL
                                         );

      public static void Main()
          {
            /* Create delegate objects */
            E04CBF_FUNCT_DELEGATE FUNCT = new E04CBF_FUNCT_DELEGATE(myfunct);
            E04CBF_MONIT_DELEGATE MONIT = new E04CBF_MONIT_DELEGATE(mymonit);
            int n = 2;
            double [] x = {-1.0, 1.0};
            double f = 99.0;
            double tolx = 1.0e-6;
            double tolf = 1.0e-6;
            int [] iuser = new int[1];
            double [] ruser = new double[1];
            int ifail = 1;
            int maxcal = 100;
            int imonit;

            imonit = 0; /* Set to 1 to turn on monitoring information, 0 for off */
            iuser[0] = imonit; /* The iuser array can contain whatever we like, so
                                  we use it to pass the value of imonit */

            if (RuntimeInformation.IsOSPlatform(OSPlatform.Windows))
            {
                E04CBF(ref n, x, ref f, ref tolf, ref tolx, FUNCT, MONIT, ref maxcal, iuser, ruser, ref ifail);
            }
            else if (RuntimeInformation.IsOSPlatform(OSPlatform.Linux))
            {
                E04CBF_(ref n, x, ref f, ref tolf, ref tolx, FUNCT, MONIT, ref maxcal, iuser, ruser, ref ifail);
            }
            else
            {
                Console.WriteLine("Unsupported OS Platform");
                return;
            }
            if (ifail != 0)
            {
                Console.WriteLine("ifail = {0}", ifail);
                // throw exception
            }
            Console.WriteLine("Final value = {0, 7:f4}\n at the point {1, 7:f4}, {2, 7:f4}", f, x[0], x[1]);
          }

        public static void myfunct(ref int n, IntPtr xc_ptr, ref double fc,
                                   IntPtr iuser_ptr /* int [] iuser*/, IntPtr ruser_ptr /* double [] ruser*/)
          {
            double [] xc = new double[n];
            Marshal.Copy( xc_ptr, xc, 0, n );
            fc = Math.Exp(xc[0]) * (xc[0] * 4.0 * (xc[0] + xc[1]) +
                               xc[1] * 2.0 * (xc[1] + 1.0) + 1.0);
          }

        public static void mymonit(ref double fmin, ref double fmax, IntPtr sim_ptr /* double [,] sim */,
                                   ref int n, ref int ncall, ref double serror, ref double vratio,
                                   IntPtr iuser_ptr /* int [] iuser*/, IntPtr ruser_ptr /* double [] ruser*/)
          {
            int ldsim = n + 1;
            double [] sim = new double[ldsim*n];
            Marshal.Copy( sim_ptr, sim, 0, ldsim*n );/* Marshal.copy does not allow the type of sim to be [,] */
            int [] iuser = new int[1];
            Marshal.Copy( iuser_ptr, iuser, 0, 1 );
            int imonit = iuser[0];

            if (imonit != 0 )
            {
                    Console.WriteLine("After {0} calls, the Simplex is ", ncall);
                    for (int i=0; i<ldsim; i++)
                    {
                        for (int j=0; j<n; j++)
                        {
                                /* Note the expression required to access the elements of sim */
                                Console.Write("{0, 12:F4}", sim[j*ldsim+i]);
                        }
                        Console.WriteLine("");
                    }
            }
          }
  }
}
