using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.InteropServices;

/* This example calls either a Windows or Linux version of the NAG Library based on
 * which OS the example is being executed on*/

namespace E02BAF
{
    class Program
    {

        /* NAG routine declarations copied from flcsdnet64.cs & flcsdnet64_linux.cs*/

        /* Both Windows and Linux routine declarations included as the dll is not 
         * loaded unless the routine is called explicitly.
         * The Linux declaration has an underscore appended to differentiate it*/

        [DllImport("NLW6I31DE_nag.dll")] /* Import from DLL, the C# compiler provides a rudimentary check of the signature */
        public static extern void E02BAF(
                                         ref int M,
                                         ref int NCAP7,
                                         [In] double[] X,
                                         [In] double[] Y,
                                         [In] double[] W,
                                         [In,Out] double[] LAMDA,
                                         [Out] double[] WORK1,
                                         [Out] double[] WORK2,
                                         [Out] double[] C,
                                         ref double SS,
                                         ref int IFAIL
                                         );

      [DllImport("libnag_nag.so", EntryPoint="e02baf_")]
        public static extern void E02BAF_(
                                         ref int M,
                                         ref int NCAP7,
                                         [In] double[] X,
                                         [In] double[] Y,
                                         [In] double[] W,
                                         [In,Out] double[] LAMDA,
                                         [Out] double[] WORK1,
                                         [Out] double[] WORK2,
                                         [Out] double[] C,
                                         ref double SS,
                                         ref int IFAIL
                                         );

      [DllImport("NLW6I31DE_nag.dll")]
        public static extern void E02BBF(
                                         ref int NCAP7,
                                         [In] double[] LAMDA,
                                         [In] double[] C,
                                         ref double X,
                                         ref double S,
                                         ref int IFAIL
                                         );

      [DllImport("libnag_nag.so", EntryPoint="e02bbf_")]
        public static extern void E02BBF_(
                                         ref int NCAP7,
                                         [In] double[] LAMDA,
                                         [In] double[] C,
                                         ref double X,
                                         ref double S,
                                         ref int IFAIL
                                         );

        static void Main(string[] args)
        {
            double[] x = { 0.0, 1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0, 11.0, 12.0, 13.0 };
            double[] y = { 0, 0.841470985, 0.909297427, 0.141120008, -0.756802495, -0.958924275, -0.279415498, 0.656986599, 0.989358247, 0.412118485, -0.544021111, -0.999990207, -0.536572918, 0.420167037 };
            double[] w = { 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0 };
            int m = x.Length;
            int n = 2 * m - 1;
            int ncap = 4;
            int ncap7 = ncap + 7;

            /* Knots array lamda1 - note that we use 0.0 as placeholders for the non-interior knots */
            double[] lamda1 = { 0.0, 0.0, 0.0, 0.0, 4.0, 5.0, 6.0, 0.0, 0.0, 0.0, 0.0 };

            double[] work1 = new double[m];
            double[] work2 = new double[4 * ncap7];
            double[] c1 = new double[ncap7];
            double[] xx = new double[n];
            double ss = 0.0D;
            double fit = 0.0D;
            int ifail;

            ifail = 1;

             try
             {
                 if (RuntimeInformation.IsOSPlatform(OSPlatform.Windows))
                 {
                     E02BAF(ref m, ref ncap7, x, y, w, lamda1, work1, work2, c1, ref ss, ref ifail);
                 }
                 else if (RuntimeInformation.IsOSPlatform(OSPlatform.Linux))
                 {
                     E02BAF_(ref m, ref ncap7, x, y, w, lamda1, work1, work2, c1, ref ss, ref ifail);
                 }
                 switch (ifail)
                 {
                     case 1:
                         throw new Exception("The knots fail to satisfy the condition x[0] < lamda[4] <= lamda[5] <=...<= lamda[ncap7-5) < x[m-1]. Thus the knots are not in correct order or are not interior to the data interval.");
                     case 2:
                         throw new Exception("The weights are not all strictly positive.");
                     case 3:
                         throw new Exception("The values of x[r], for r = 0, 1, 2, ..., m - 1, are not in nondecreasing order.");
                     case 4:
                         throw new Exception("ncap7 < 8 (so the number of interior knots is negative) or ncap7 > mdist + 4, where mdist is the number of distinct x values in the data (so there cannot be a unique solution).");
                     case 5:
                         throw new Exception("The conditions specified by Schoenberg and Whitney (1953) fail to hold for at least one subset of the distinct data abscissae. That is, there is no subset of ncap7 - 4 strictly increasing values, x[R[0]-1],..., x[R[ncap7-5]-1], among the abscissae such that, x[R[0]-1] < lamda[0] < x[R[4]-1], x[R[1]-1] < lamda[1] < x[R[5]-1],...,x[R(ncap7-9]-1] < lamda[ncap7-9] < lamda[R[ncap7-5]-1]");
                     default:
                         if (ifail < 0)
                         {
                             throw new Exception(String.Format("E02BAF returned error code ifail = {0}", ifail));
                         }                       
                         break;
                 }
             }
             catch (Exception ex)
             {
                 Console.WriteLine(ex.Message);
             }

             //fill xx-array with values to be evaluated
             for (int i = 0; i < m; i++)
             {
                 xx[2 * i] = x[i];
                 if (i > 0)
                     xx[2 * i - 1] = 0.5 * (x[i - 1] + x[i]);
             }

             if (ifail == 0)
             {
                 Console.WriteLine("Abscissa    Spline\n");

                 for (int i = 0; i < xx.Length; i++)
                 {
                     ifail = 1;
                     try
                     {
                         if (RuntimeInformation.IsOSPlatform(OSPlatform.Windows))
                         {
                             E02BBF(ref ncap7, lamda1, c1, ref xx[i], ref fit, ref ifail);
                         }
                         else if (RuntimeInformation.IsOSPlatform(OSPlatform.Linux))
                         {
                             E02BBF_(ref ncap7, lamda1, c1, ref xx[i], ref fit, ref ifail);
                         }
                         switch (ifail)
                         {
                             case 1:
                                 throw new Exception("The parameter x does not satisfy lamda[4] <= x <= lamda(ncap7-3). In this case the value of S is set arbitrarily to zero.");
                             case 2:
                                 throw new Exception("ncap7 < 8, i.e., the number of interior knots is negative.");
                             default:
                                 if (ifail < 0)
                                 {
                                     throw new Exception(String.Format("E02BBF returned error code ifail = {0}", ifail));
                                 }                       
                                 break;
                         }

                     }
                     catch (Exception ex)
                     {
                         Console.WriteLine(ex.Message);
                         break;
                     }

                     Console.WriteLine("{0} \t {1, 10:f4}", xx[i], fit);
                 }
             }

        }
    }
}
