using System;
using System.Runtime.InteropServices; /* Provides mappings between C# and native code */


/* This example calls either a Windows or Linux version of the NAG Library based on
 * which OS the example is being executed on */

namespace NumericalAlgorithmsGroup
{
  class Quadrature
    {
        /* NAG routine declarations copied from flcsdnet64.cs & flcsdnet64_linux.cs */

        /* Both Windows and Linux routine declarations included as the dll is not 
         * loaded unless the routine is called explicitly.
         * The Linux declaration has an underscore appended to differentiate it */

        /* Delegate object, note that the scalar parameter is 'ref' */
      public delegate double D01AHF_F_DELEGATE(
                                               ref double X
                                               );

      [DllImport("NLW6I31DE_nag.dll")] /* Import from DLL, the C# compiler provides a rudimentary check of the signature */
        public static extern double D01AHF(
                                           ref double A,
                                           ref double B,
                                           ref double EPSR,
                                           ref int NPTS,
                                           ref double RELERR,
                                           D01AHF_F_DELEGATE F,
                                           ref int NLIMIT,
                                           ref int IFAIL
                                           );

      [DllImport("libnag_nag.so", EntryPoint = "d01ahf_")]
        public static extern double D01AHF_(
                                           ref double A,
                                           ref double B,
                                           ref double EPSR,
                                           ref int NPTS,
                                           ref double RELERR,
                                           D01AHF_F_DELEGATE F,
                                           ref int NLIMIT,
                                           ref int IFAIL
                                           );

      public static void Main()
      {
          double a = 0.0;
          double b = 1.0;
          double epsr = 0.00001;
          int npts = 0;
          double relerr = 0.0;
          /* Create a delegate object */
          D01AHF_F_DELEGATE FUNC = new D01AHF_F_DELEGATE(func);
          int nlimit = 0;
          int ifail = 1;
          double ans = 0.0;

          if (RuntimeInformation.IsOSPlatform(OSPlatform.Windows))
            {
              ans = D01AHF(ref a, ref b, ref epsr, ref npts, ref relerr, FUNC,
                           ref nlimit, ref ifail);
            }
          else if (RuntimeInformation.IsOSPlatform(OSPlatform.Linux))
            {
              ans = D01AHF_(ref a, ref b, ref epsr, ref npts, ref relerr, FUNC,
                            ref nlimit, ref ifail);
            }
          else
            {
              Console.WriteLine("Unsupported OS Platform");
              return;
            }

          Console.WriteLine("D01AHF Example Program Results");
          if (ifail != 0)
            {
                Console.WriteLine("ifail = {0}", ifail);
              // throw exception
            }
          else
            {
              Console.WriteLine("Integral = {0, 8:f5}", ans);
              Console.WriteLine("Estimated relative error = {0, 10:e2}", relerr);
              Console.WriteLine("Number of function evaluations = {0}", npts);
            }
      }

      public static double func(ref double x)
      {
          double retval;
          retval = 4.0 / (1.0 + x*x);
          return retval;
      }
    }
}
